<?php

namespace App\Http\Controllers\Api\Auth;

use App\Models\User;
use App\Mail\WelcomeMail;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\NewAccessToken;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Config;

class AuthController extends Controller
{
    public function signUp(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'password' => 'required|min:6|max:100',
            'email' => 'required|string',
            'fcm_token' => 'required|string',
        ]);

        $code = random_int(100000, 999999);
        $expire = now()->addMinutes(env('OTP_VISIBILITY_TIME') ?? 3);
        $data = [
            'code' => $code,
            'name' => $request->name,
        ];

        $user = User::where('email', $request->email)->first();
        if ($user && $user->business_id) {
            return response()->json([
                'message' => 'This email is already exists.',
            ], 406);
        }

        if (env('MAIL_USERNAME')) {
            if (env('QUEUE_MAIL')) {
                Mail::to($request->email)->queue(new WelcomeMail($data));
            } else {
                Mail::to($request->email)->send(new WelcomeMail($data));
            }
        } else {
            return response()->json([
                'message' => __('Mail service is not configured. Please contact your administrator.'),
            ], 406);
        }

        $user = User::updateOrCreate(['email' => $request->email], $request->except('password') + [
                    'remember_token' => $code,
                    'email_verified_at' => $expire,
                    'password' => Hash::make($request->password),
                ]);
        return response()->json([
            'message' => 'An otp code has been sent to your email. Please check and confirm.',
            'data' => $user,
        ]);
    }

    public function submitOtp(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'otp' => 'required|min:4|max:15',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return response()->json([
                'status' => 404,
                'message' => __('User not found'),
            ], 404);
        }

        if ($user->remember_token == $request->otp) {
            if ($user->email_verified_at > now()) {

                Auth::login($user);
                $is_setup = $user->business_id ? true : false;
                $token = $user->createToken('createToken')->plainTextToken;
                $accessToken = $user->createToken('createToken');
                $this->setAccessTokenExpiration($accessToken);

                $user->update([
                    'remember_token' => NULL,
                    'email_verified_at' => now(),
                ]);

                return response()->json([
                    'message' => 'Logged In successfully!',
                    'is_setup' => $is_setup,
                    'token' => $token,
                ]);

            } else {
                return response()->json([
                    'error' => __('The verification otp has been expired.')
                ], 400);
            }
        } else {
            return response()->json([
                'error' => __('Invalid otp.')
            ], 404);
        }
    }

    public function login(Request $request)
    {
        $request->validate([
            'password' => 'required',
            'email' => 'required|string',
            'fcm_token' => 'nullable|string',
        ]);

        if (auth()->attempt($request->only('email', 'password'))) {
            $user = auth()->user();

            $business = $user->business;
            if ($business && !$business->status) {
                Auth::guard('web')->logout();
                return response()->json([
                    'message' => 'Your business is inactive. Please contact your administrator.'
                ], 406);
            }

            if (!in_array($user->role, ['staff', 'shop-owner', 'chef', 'kitchen'])) {
                return response()->json([
                    'message' => 'You can not login as ' . $user->role . ' from the app!'
                ], 406);
            }

            if ($user->remember_token && !$user->business_id) { // If user didn't verify email

                $code = random_int(100000, 999999);
                $expire = now()->addMinutes(env('OTP_VISIBILITY_TIME') ?? 3);
                $data = [
                    'code' => $code,
                    'name' => $request->name,
                ];

                if (env('MAIL_USERNAME')) {
                    if (env('QUEUE_MAIL')) {
                        Mail::to($request->email)->queue(new WelcomeMail($data));
                    } else {
                        Mail::to($request->email)->send(new WelcomeMail($data));
                    }
                } else {
                    return response()->json([
                        'message' => __('Mail service is not configured. Please contact your administrator.'),
                    ], 406);
                }

                User::where('email', $request->email)->first()->update(['remember_token' => $code, 'email_verified_at' => $expire]);

                return response()->json([
                    'message' => 'An otp code has been sent to your email. Please check and confirm.',
                ], 201);
            }

            $fcmToken = $request->fcm_token;

            if ($fcmToken) {
                $tokens = $user->fcm_token ?? [];
                if (!is_array($tokens)) {
                    $tokens = [];
                }

                $tokens = array_values(
                    array_unique(
                        array_merge($tokens, [$fcmToken])
                    )
                );

                $user->update([
                    'fcm_token' => $tokens,
                ]);
            }
            return response()->json([
                'message' => 'User login successfully!',
                'data' => [
                    'message' => 'Logged In successfully!',
                    'is_setup' => $user->business_id ? true : false,
                    'token' => $user->createToken('createToken')->plainTextToken,
                ],
            ]);
        } else {
            return response()->json([
                'message' => 'Invalid email or password!'
            ], 404);
        }
    }

    protected function setAccessTokenExpiration(NewAccessToken $accessToken)
    {
        $expiration = now()->addMinutes(Config::get('sanctum.expiration'));

        DB::table('personal_access_tokens')
            ->where('id', $accessToken->accessToken->id)
            ->update(['expires_at' => $expiration]);
    }

    public function signOut() : JsonResponse
    {
        if (auth()->user()->tokens()) {
            auth()->user()->tokens()->delete();

            return response()->json([
                'message' => __('Sign out successfully'),
            ]);
        } else {
            return response()->json([
                'message' => __('Unauthorized'),
            ], 401);
        }
    }

    public function refreshToken()
    {
        if (auth()->user()->tokens()) {

            auth()->user()->currentAccessToken()->delete();
            $data['token'] = auth()->user()->createToken('createToken')->plainTextToken;
            return response()->json($data);

        } else {
            return response()->json([
                'message' => __('Unauthorized'),
            ], 401);
        }
    }

    public function resendOtp(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
        ]);

        $code = random_int(100000, 999999);
        $expire = now()->addMinutes(env('OTP_VISIBILITY_TIME') ?? 3);
        $data = [
            'code' => $code,
            'name' => $request->name,
        ];

        if (env('MAIL_USERNAME')) {
            if (env('QUEUE_MAIL')) {
                Mail::to($request->email)->queue(new WelcomeMail($data));
            } else {
                Mail::to($request->email)->send(new WelcomeMail($data));
            }
        } else {
            return response()->json([
                'message' => __('Mail service is not configured. Please contact your administrator.'),
            ], 406);
        }

        User::where('email', $request->email)->first()->update(['remember_token' => $code, 'email_verified_at' => $expire]);

        return response()->json([
            'message' => 'An otp code has been sent to your email. Please check and confirm.',
        ]);
    }
}
