<?php

namespace App\Http\Controllers\Api;

use App\Models\Expense;
use App\Models\Business;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class AcnooExpenseController extends Controller
{
    public function index()
    {
        $data = Expense::with('category:id,categoryName')
        ->where('business_id', auth()->user()->business_id)
        ->when(request('expense_category_id'), function($query) {
            $query->where('expense_category_id', request('expense_category_id'));
        })
        ->when(request('search'), function($query) {
            $query->where('expanseFor', 'like', '%'.request('search'). '%')
                  ->orWhere('amount', 'like', '%'.request('search'). '%')
                  ->orWhereHas('category', function($query) {
                    $query->where('categoryName', 'like', '%'.request('search'). '%');
                  });
          })
        ->latest()
        ->paginate(10);

        return response()->json([
            'message' => __('Data fetched successfully.'),
            'data' => $data,
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0|max:99999999.99',
            'expense_category_id' => 'required|exists:expense_categories,id',
        ]);

        Business::findOrFail(auth()->user()->business_id)->decrement('remainingShopBalance', $request->amount);

        $data = Expense::create($request->except('status') + [
                    'user_id' => auth()->id(),
                    'business_id' => auth()->user()->business_id,
                ]);

        return response()->json([
            'message' => __('Expense saved successfully.'),
            'data' => $data,
        ]);
    }

    public function update(Request $request, string $id) {

        $request->validate([
            'amount' => 'required|numeric|min:0|max:99999999.99',
            'expense_category_id' => 'required|exists:expense_categories,id',
        ]);

        $expense = Expense::findOrFail($id);
        $business = Business::findOrFail(auth()->user()->business_id);

        $amountDifference = $request->amount - $expense->amount;

        $business->increment('remainingShopBalance', $amountDifference);


        $expense->update($request->except('user_id', 'business_id') + [
            'user_id' => auth()->id(),
            'business_id' => auth()->user()->business_id,

        ]);

        return response()->json([
            'message' => __('Expense updated successfully.'),
            'data' => $expense,
        ]);
    }

    public function destroy(string $id) {

        $expense = Expense::findOrFail($id);
        $expense->delete();

        return response()->json([
            'message' => __('Expense deleted successfully.'),
        ]);
    }
}
